/*************************************************************************
 *
 * Hitchhiker's Guide to the IBM PE 
 * Optimized Ray Tracing Program
 * Chapter 4 - So Long and Thanks For All The Fish
 *
 * To compile:
 * mpcc -pg -o rtrace_fast rtrace_fast.c
 *
 * Description: 
 * This is a sample program that partitions N tasks into
 * two groups, a collect node and N - 1 compute nodes.
 * The responsibility of the collect node is to collect the data
 * generated by the compute nodes. The compute nodes send the 
 * results of their work to the collect node for collection.
 *
 *************************************************************************/

#include <mpi.h>

#define PIXEL_WIDTH	50
#define PIXEL_HEIGHT	50

int	First_Line = 0;

void main(int argc, char *argv[])
{ 
  int    	numtask;
  int	taskid;
 
  /* Find out number of tasks/nodes. */
  MPI_Init( &argc, &argv);
  MPI_Comm_size( MPI_COMM_WORLD, &numtask);
  MPI_Comm_rank( MPI_COMM_WORLD, &taskid);

  /* Task 0 is the coordinator and collects the processed pixels */
  /* All the other tasks process the pixels                      */
  if ( taskid == 0 )
    collect_pixels(taskid, numtask);
  else
    compute_pixels(taskid, numtask);

  printf("Task %d waiting to complete.\n", taskid);
  /* Wait for everybody to complete */
  MPI_Barrier(MPI_COMM_WORLD);
  printf("Task %d complete.\n",taskid);
  MPI_Finalize();
  exit(); 
}

/* In a real implementation, this routine would process the pixel */
/* in some manner and send back the processed pixel along with its*/
/* location.  Since we're not processing the pixel, all we do is  */
/* send back the location                                         */
compute_pixels(int taskid, int numtask)
{
  int		offset;
  int		row, col;
  int		*pixel_row;
  int		buff_width;
  MPI_Status	stat;

  printf("Compute #%d: checking in\n", taskid);

  First_Line = (taskid - 1);            /* First n-1 rows are assigned */
					/* to processing tasks         */
  offset = numtask - 1;                 /* Each task skips over rows   */
					/* processed by other tasks    */

  pixel_row = (int *) malloc( PIXEL_WIDTH * sizeof(int) * 2);

  for (row = First_Line; row < PIXEL_HEIGHT; row += offset)
    {
      buff_width = 2 * PIXEL_WIDTH;
      for ( col = 0; col < buff_width; col += 2)
	{
	  pixel_row [col] = row;
	  pixel_row [col + 1] = col / 2;
	}
      printf(" sending row %d\n", row);
      MPI_Send(pixel_row, buff_width, MPI_INT, 0, 0, MPI_COMM_WORLD);
    }

  free( pixel_row);
  printf("Compute #%d: done sending. ", taskid);
  return;
}

/* This routine collects the pixels.  In a real implementation, */
/* after receiving the pixel data, the routine would look at the*/
/* location information that came back with the pixel and move  */
/* the pixel into the appropriate place in the working buffer   */
/* Since we aren't doing anything with the pixel data, we don't */
/* bother and each message overwrites the previous one          */
collect_pixels(int taskid, int numtask)
{
  int		*pixel_row;
  int		buff_width;
  int		col;
  MPI_Status	stat;
  int     	mx = PIXEL_HEIGHT;

  printf("Control #%d: No. of nodes used is %d\n",taskid,numtask);
  printf("Control: expect to receive %d messages\n", mx);

  pixel_row = (int *) malloc( PIXEL_WIDTH * sizeof(int) * 2);
  buff_width = 2 * PIXEL_WIDTH;
  while (mx > 0)
    {
      MPI_Recv(pixel_row, buff_width, MPI_INT, MPI_ANY_SOURCE, 
	       MPI_ANY_TAG, MPI_COMM_WORLD, &stat);
      printf("received row %d\n",mx);
      mx--;
    }
  free(pixel_row);
  printf("Control node #%d: done receiving. ",taskid);
  return;
}
